function varargout = pore_processing_gui_call(src,evt,callbackstr,varargin)
% Callback function for pore processing gui


fig = getfig(src);
if isempty(fig)
    fig = gcf;
end


switch callbackstr
%% init    
    case 'init'
        % Initialize gui
        set(fig,'Name','Pore Image Processor','NumberTitle','off');
        set(fig,'KeyPressFcn',{@pore_processing_gui_call,'key'}); 
        pos = get(fig,'Position');
        set(fig,'Position',[1,pos(2:4)]); % move to left edge of screen
        
        % Initialize tab selection
        cb = get(getappdata(fig,'mode_p'),'Callback');
        feval(cb{1},[],[],cb{2:end});
        
        % Mark results as non-current
        pore_processing_gui_call([],[],'set_results_current',false);
        
        pore_processing_gui_call([],[],'update se'); %initialize structuring element
        status('GUI initialized');
%% key
    case 'key'
        %%% NOTE: When zooming is turned on, the zooming mode overrides the
        %%% figure's normal KeyPressFcn (they are restored when zooming is
        %%% turned off), so these key presses will not arrive here when 
        %%% zooming is on.  However, I have slightly modified the MATLAB
        %%% zoom function so that pressing the 'z' key turns it back off.
        %%% If you are not using this modified zoom function, then you'll
        %%% have to turn off zooming in some other way (e.g. clicking the
        %%% maginfying glass) before the keys will work here.
        if ~isempty(evt)
            key = evt.Character;
        else
            key = varargin{1};
        end
        switch key
            case 'z'
                % toggle zoom state
                zoom;
            case 'b'
                % toggle pore boundaries
                view_hs = getappdata(fig,'view_handles');
                val = get(view_hs.showbound,'Value');
                set(view_hs.showbound,'Value',~val);
                pore_processing_gui_call([],[],'pore boundaries');
            case 'r'
                % recalculate results
                pore_processing_gui_call([],[],'update results');
            case 's'
                % Warn if results out of date
                if ~ getappdata(fig,'results_current_flag')
                    % results not current
                    response = questdlg('Warning, you have made alterations to the processing since the last time the results were updated. It is recommended that you update the results before saving them.',...
                        'Results Need Updating','Save Anyway','Cancel','Cancel');
                    switch response
                        case {'','Cancel'}
                            % cancel saving
                            return
                    end
                end
                    
                % save results
                results = getappdata(fig,'results'); %#ok<NASGU>
                [fname,pname] = uiputfile('*_r.mat','Save Results Structure As...');
                if ischar(fname)
                    if ~strcmp(fname(end-5:end),'_r.mat')
                        fname = [fname '_r.mat'];
                    end
                    save([pname fname],'results','-mat');
                    status(['Results saved to ' fname]);
                end
            case 'i'
                % capture image and save
                f = getframe(getappdata(fig,'image_ax'));
                [fname,pname] = uiputfile('*.jpg','Save Image As...');
                if ischar(fname)
                    if ~strcmp(fname(end-3:end),'.jpg')
                        fname = [fname '.jpg'];
                    end

                    imwrite(f.cdata,[pname fname],'JPG');
                    status(['Current axes image saved to ' fname]);
                end
                
        end
%% browse for orig        
    case 'browse for orig'
        % Choose original image file
        [filename,pathname] = uigetfile({'*.jpg;*.jpeg','JPEG Image Files';...
            '*.bmp;*.gif;*.png;*.tif;*.tiff',  'Other Image Files'; ...
            '*.*',  'All Files (*.*)'}, ...
            'Choose Original Image File');


        if ischar(filename)
            orig_hs = getappdata(fig,'original_hs');
            set(orig_hs.filename,'String',[pathname filename]);
            pore_processing_gui_call([],[],'load orig');
        end
%% load orig        
    case 'load orig'
        % load indicated image file
        orig_hs = getappdata(fig,'original_hs');
        imname = get(orig_hs.filename,'String');
        if ~exist(imname,'file')
            errordlg('Image filename and path are not valid!','Error: Invalid filename');
            return
        end
        status('Loading new original image...');
        im = imread(imname);
        if size(im,3)==3
            %flatten image
            im = im(:,:,1); % Convert to grayscale
        end
        % Put the image on the axes uncropped, and reset any logging
        im_ax = getappdata(fig,'image_ax');
        imh = imagesc(im,'Parent',im_ax,'ButtonDownFcn',{@pore_processing_gui_call,'image click'});
        drawnow;
        colormap(gray); axis(im_ax,'image');
        setappdata(fig,'image_h',imh);
        drawnow; zoom reset; % reset zoom point with new image
        setappdata(fig,'orig',im);
        setappdata(fig,'bgc',im); % if no background correction, the bgc is just the original
        % Clear other images
        setappdata(fig,'bw',[]);
        % Mark results as non-current
        pore_processing_gui_call([],[],'set_results_current',false);
        % Change view radio button
        pore_processing_gui_call([],[],'change view','orig');
        
        status('New original image loaded');
        addlog('load',imname);
%% crop        
    case 'crop'
        crop_hs = getappdata(fig,'crop_handles');
        sel_obj = get(crop_hs.rbg,'SelectedObject');
        ax = getappdata(fig,'image_ax');
        status('Cropping image...');
        if isempty(sel_obj)
            errordlg('No cropping method selected!','Error');
            status('Cropping aborted');
        elseif sel_obj==crop_hs.curax
            % Crop to current axis limits
            axlim = axis(ax);
            status('Cropping image to current axis limits...');
        elseif sel_obj==crop_hs.manual
            % Crop to manual limits
            xmin = str2double(get(crop_hs.limL,'String'));
            xmax = str2double(get(crop_hs.limR,'String'));
            ymin = str2double(get(crop_hs.limB,'String'));
            ymax = str2double(get(crop_hs.limT,'String'));
            axlim = round([xmin,xmax,ymin,ymax]);        
            status('Cropping image to manual limits...')
        end
        % crop to given limits
        im = getappdata(fig,'orig');
        axlim = [ceil(axlim(1)),floor(axlim(2)),ceil(axlim(3)),floor(axlim(4))];
        cropped_im = im(axlim(3):axlim(4),axlim(1):axlim(2));
        imh = imagesc(cropped_im,'Parent',ax,'ButtonDownFcn',{@pore_processing_gui_call,'image click'});
        drawnow; 
        colormap(gray); axis(ax,'image');
        zoom reset;
        pore_processing_gui_call([],[],'set threshlimits');% set threshold max and min
        setappdata(fig,'image_h',imh);
        setappdata(fig,'orig',cropped_im); % the uncropped original is not saved
        setappdata(fig,'bgc',cropped_im); % if no background correction, bgc is just orig
        setappdata(fig,'bw',[]);
        % Mark results as non-current
        pore_processing_gui_call([],[],'set_results_current',false);        
        % Change view radio button
        pore_processing_gui_call([],[],'change view','orig',true);

        status('Image cropped');
        addlog('crop',axlim);
%% correct bg        
    case 'correct bg'
        % Correct background using an averaging filter and image
        % subtraction
        bgc_hs = getappdata(fig,'bgc_handles');
        method = 'average'; %much faster than disk
        filtersize = str2double(get(bgc_hs.filtersize,'String'));
        pore_processing_gui_call([],[],'showfilter'); % reset filter display if necessary
        status('Median filtering to remove salt and pepper noise...');
        bgc_im = medfilt2(getappdata(fig,'orig')); % median filtering gets rid of salt and pepper noise
        status('Correcting background... (this can take a few minutes)')
        bgc_im = bgfix(bgc_im,method,filtersize);
        set(getappdata(fig,'image_h'),'CData',bgc_im);
        setappdata(fig,'bgc',bgc_im);
        setappdata(fig,'bw',[]);
        drawnow;
        % Mark results as non-current
        pore_processing_gui_call([],[],'set_results_current',false);        
        % Change view radio button
        pore_processing_gui_call([],[],'change view','bgc',true);
        
        pore_processing_gui_call([],[],'set threshlimits');% set threshold max and min
        status('Background corrected');
        addlog('bgc',filtersize);
%% showfilter        
    case 'showfilter'
        % toggle whether the averaging filter is shown on the image.
        % Should be transparent and draggable, maybe green.
        bgc_hs = getappdata(fig,'bgc_handles');
        filtersize = str2double(get(bgc_hs.filtersize,'String'));
        showfilter = get(bgc_hs.showfilter,'Value');
        filter_h = getappdata(fig,'filter_h');
        if showfilter
            % Plot the filter in the upper left hand corner, make it
            % draggable (later)
            if ~isempty(filter_h) && ishandle(filter_h)
                delete(filter_h);
            end
            filter_h = patch([1,1,filtersize,filtersize],[1,filtersize,filtersize,1],'g','EdgeColor','none','FaceAlpha',.5,'Parent',getappdata(fig,'image_ax'));
            setappdata(fig,'filter_h',filter_h);
            drawnow;
            status('Averaging filter shown');
        else
            if ~isempty(filter_h) && ishandle(filter_h)               
                delete(filter_h);
                status('Averaging filter removed');
            end
            setappdata(fig,'filter_h',[]);
            
        end
%% thresh        
    case 'thresh'
        % threshold current bgc
        thresh_hs = getappdata(fig,'thresh_handles');
        thresh = str2double(get(thresh_hs.thresh,'String'));
        if thresh>str2double(get(thresh_hs.max,'String')) || thresh<str2double(get(thresh_hs.min,'String'))
            errordlg('Threshold value must be between the minimum and maximum values in the image!','Error: Threshold out of bounds');
            return
        end
        status('Applying threshold...');
        bgc = getappdata(fig,'bgc');
        thresh_im = bgc > thresh;
        set(getappdata(fig,'image_h'),'CData',thresh_im);
        drawnow;
        setappdata(fig,'bw',thresh_im);
        % Mark results as non-current
        pore_processing_gui_call([],[],'set_results_current',false); 
        % Change view radio button
        pore_processing_gui_call([],[],'change view','bw',true);

        status('Threshold applied');
%% threshlimits        
    case 'set threshlimits'
        bgc = getappdata(fig,'bgc');
        threshmax = max(bgc(:));
        threshmin = min(bgc(:));
        thresh_hs = getappdata(fig,'thresh_handles');
        set(thresh_hs.min,'String',num2str(threshmin));
        set(thresh_hs.max,'String',num2str(threshmax));
%% change mode
    case 'change mode'
        mode_p = varargin{1};
        panels = varargin{2};
        layouts = varargin{3};
        param = varargin{4};
        % determine which mode was chosen
        % make that panel visible
        % make others invisible
        str = get(mode_p,'String');
        chosen_str = str{get(mode_p,'Value')};
        switch chosen_str
            case 'Load, Crop, and Correct'
                mode_str = 'load';
            case 'Binary Operations'
                mode_str = 'binop';
            case 'Algorithm Log'
                mode_str = 'log';
            case 'Results'
                mode_str = 'res';
            otherwise
                error('Someone must have changed the strings in the mode selection popupmenu. They need to match up with these choices')
        end
        set([panels.load,panels.binop,panels.log,panels.res],'Visible','off'); % make them all invisible
        set(panels.(mode_str),'Visible','on'); % make the one you want visible
        gui_position(layouts.(mode_str),param); % position everything
        set(fig,'ResizeFcn',{@gui_position_wrapper,layouts.(mode_str),param}) % make the resize fucntion use the right layour
%% update se        
    case 'update se'
        se_hs = getappdata(fig,'se_handles');
        sel = get(se_hs.rbg,'SelectedObject');
        if isempty(sel)
            % nothing selected, just return
            return
        elseif sel==se_hs.disk_r
            % use disk parameter
            rad = str2double(get(se_hs.disk_e,'String'));
            se = strel('disk',rad);
        elseif sel==se_hs.square_r
            %use square parameter
            side = str2double(get(se_hs.square_e,'String'));
            se = strel('square',side);
        elseif sel==se_hs.custom_r
            % Launch custom SE generation window
            %% not implemented yet...
            return
        end
        setappdata(fig,'se',se);
        % Plot the SE
        nhood = getnhood(se);
        se_im = zeros(size(nhood,1)+2,size(nhood,2)+2);
        se_im(2:end-1,2:end-1) = nhood;
        imagesc(se_im,'Parent',se_hs.ax); colormap(gray); axis(se_hs.ax,'image'); axis(se_hs.ax,'off');
        status('Structuring element updated');
%% binops        
    case {'erode','dilate','open','close','majority'}
        se = getappdata(fig,'se');
        bw = getappdata(fig,'bw');
        switch callbackstr
            case 'erode'
                status('Eroding image...')
                newbw = imerode(bw,se);
                status('Image eroded');
            case 'dilate'
                status('Dilating image...')
                newbw = imdilate(bw,se);
                status('Image dilated');
            case 'open'
                status('Opening image...')
                newbw = imopen(bw,se);
                status('Image opened');
            case 'close'
                status('Closing image...')
                newbw = imclose(bw,se);
                status('Image closed');
            case 'majority'
                status('Applying majority operation until image stops changing...')
                newbw = bwmorph(bw,'majority',Inf);
                status('Majority operation applied');
        end
        set(getappdata(fig,'image_h'),'CData',newbw);
        setappdata(fig,'bw',newbw);
        % Mark results as non-current
        pore_processing_gui_call([],[],'set_results_current',false);        
                        
        
%     case 'savelog'
%         log_h = getappdata(fig,'log_handle');
%     case 'loadlog'
%         log_h = getappdata(fig,'log_handle');
        
%% image click
    case 'image click'
        % user clicked on image
        % Find clicked point
        xyz = get(getappdata(fig,'image_ax'),'CurrentPoint');
        xy = round(xyz(1,1:2));
        % Find pore at that point
        results = getappdata(fig,'results');
       
        if isempty(results)
            return
        end
        L = results.L;
        pore_num = L(xy(2),xy(1));
        if ~pore_num
            % not on a pore
            return;
        end
        
        switch get(fig,'SelectionType')
            case 'normal'
                pore_processing_gui_call([],[],'select pore',pore_num);
            case 'open'
                pore_processing_gui_call([],[],'remove pore',pore_num);
        end
%% porebound click        
    case 'porebound click'
        % user clicked on pore boundary
        pore_num = varargin{1};
        switch get(fig,'SelectionType')
            case 'normal'
                pore_processing_gui_call([],[],'select pore',pore_num);
            case 'open'
                pore_processing_gui_call([],[],'remove pore',pore_num);
        end
%% select pore
    case 'select pore'
        pore_num = varargin{1};
        results = getappdata(fig,'results');
        % Highlight selected pore and eliminate old selected pore 
        pore = results.pores(logical([results.pores.pore_num]==pore_num));
        sel_bound = pore.boundary;
        %sel_bound = results.B{pore_num}; %old way
        selpore_h(1) = line(sel_bound(:,2),sel_bound(:,1),'Color',rgb('purple'),...
            'LineWidth',3,'Parent',getappdata(fig,'image_ax'),...
            'ButtonDownFcn',{@pore_processing_gui_call,'porebound click',pore_num});
        selpore_h(2) = line(sel_bound(:,2),sel_bound(:,1),'Color','w',...
            'LineWidth',1,'Parent',getappdata(fig,'image_ax'),...
            'ButtonDownFcn',{@pore_processing_gui_call,'porebound click',pore_num});

        old_selpore = getappdata(fig,'selpore_h');
        if any(ishandle(old_selpore))
            delete(old_selpore);
        end
        setappdata(fig,'selpore_h',selpore_h);
        status('Selected new pore');
        % Change results table
        res_hs = getappdata(fig,'results_handles');
        table = res_hs.table;
        if isempty(results.pixpernm)
            % calibrated = false;
            pixpernm = 1;
        else
            % calibrated = true;
            pixpernm = results.pixpernm;
        end

        heading_strings = table.heading_strings;
        for i = 1:length(heading_strings)
            switch heading_strings{i}
                case '#'
                    str = num2str(pore_num);
                case 'Diameter'
                    val = pore.EquivDiameter/pixpernm;
                    str = sprintf('%.1f',val);
                case 'Area'
                    val = pore.Area/pixpernm^2;
                    str = sprintf('%.1f',val);
                case 'Major Axis'
                    val = pore.MajorAxisLength/pixpernm;
                    str = sprintf('%.1f',val);
                case 'Minor Axis'
                    val = pore.MinorAxisLength/pixpernm;
                    str = sprintf('%.1f',val);
                case 'Avg Intensity'
                    val = pore.avg_intensity;
                    str = sprintf('%.1f',val);
                case 'Roundness'
                    val = pore.RoundnessIndex;
                    str = sprintf('%.2f',val);
                otherwise
                    str = '-';
            end     
            set(table.selpore_hs(i),'String',str);
        end
%% remove pore        
    case 'remove pore'
        % removes selected pore (from double click or remove button)
        %disp('-------------'); % to make separations between runs more obvious
        %disp('Trying to remove pore');
        results = getappdata(fig,'results');  %disp('getting results');
        if isempty(varargin)
            res_hs = getappdata(fig,'results_handles'); %disp('getting results handles');
            ind = strcmp('#',res_hs.table.heading_strings); %disp('getting table index');% the index of the table headings which has the pore number
            pore_num = str2double(get(res_hs.table.selpore_hs(ind),'String')); %disp('finding pore number');
        else
            pore_num = varargin{1}; %disp('pore number from input');
        end
        % Remove pore_num from results
        all_pore_nums = [results.pores.pore_num];
        remove_mask = ismember(all_pore_nums,pore_num);
        results.pores(remove_mask) = [];
        
        % Remove from label image (BW) and remove outline? 
        L = results.L; %disp('getting label image');
        for i = 1:length(pore_num)
            L(L==pore_num(i)) = 0; %disp('removing pore from label image');            
        end
        bw_im = L>0; %disp('converting to binary image, used to give a warning')
        results.L = L;
        setappdata(fig,'results',results); % this isn't quite kosher as it makes the 
                                   % Label image inconsistent with the
                                   % other results in the structure, but
                                   % the process means the whole structure
                                   % is out of date, so these results
                                   % shouldn't be being saved anyway.
        setappdata(fig,'bw',bw_im); %disp('saving new binary image to figure');
        
        % Mark results as non-current
        pore_processing_gui_call([],[],'set_results_current',false);
        pore_processing_gui_call([],[],'pore boundaries'); %remove eliminated pore boundary(s)
        %pore_processing_gui_call([],[],'update results'); %disp('updating results');
        pore_processing_gui_call([],[],'change view','bgc',true);
        if length(pore_num)==1
            status(['Pore #' num2str(pore_num) ' removed']); %disp('showing status message');
        else
            status([num2str(length(pore_num)) ' pores removed']);
        end
        
        
%% update results        
    case 'update results'
        % Calculate results based on most recent bw image, and then display
        % in results section (histogram, summary, and table), using
        % calibration if so directed
        results = getappdata(fig,'results');
        res_hs = getappdata(fig,'results_handles');
        bw = getappdata(fig,'bw');
        bgc_im = getappdata(fig,'bgc');
        if isempty(bw)
            errordlg('A black and white image must have been generated to calculate results.','Error: No B&W Image');
            return
        end
        status('Calculating boundaries...');
        [B,L] = bwboundaries(bw,'noholes');
        % find edge pore label numbers
        edge_pore_nums = unique([L(1,:),L(end,:),L(:,1)',L(:,end)']);
        status('Calculating pore properties...');
        results.pores = regionprops(L,'EquivDiameter','Area','MajorAxisLength','MinorAxisLength','Centroid','PixelIdxList');
        % Add additional pore properties
        for pore_num = 1:length(results.pores)
            % add pore number 
            results.pores(pore_num).pore_num = pore_num;
            % add boundary info
            results.pores(pore_num).boundary = B{pore_num};
            % add edge flag
            if ismember(pore_num,edge_pore_nums)
                results.pores(pore_num).non_edge_flag = 0;
            else
                results.pores(pore_num).non_edge_flag = 1;
            end
            % add mean pore intensity
            results.pores(pore_num).avg_intensity = mean(bgc_im(results.pores(pore_num).PixelIdxList));
            % add Roundness Index
            results.pores(pore_num).RoundnessIndex = results.pores(pore_num).MinorAxisLength / results.pores(pore_num).MajorAxisLength;
        end
        % Store label matrix
        results.L = L;
        pixpernm = getappdata(fig,'pixpernm'); % empty if undefined or box unchecked
        % Calculate results
        status('Calculating statistics...');   
        if isempty(pixpernm)
            calibrated = false;
            statname = 'stats'; %use pixel stats
            if isfield(results,'nmstats')
                results.nmstats = [];
            end
        else
            calibrated = true;
            statname = 'nmstats'; %use nm stats
            results = calc_stats(results,[]); %also calc pixel stats
        end
        results = calc_stats(results,pixpernm); % calculate nm stats if calibrated
        setappdata(fig,'results',results);
        
        % Now update histogram, summary, and table
        status('Updating histogram of pore diameters...');
        num_bins = 50; % number of histogram bins
        nonedge_pores = find_nonedge_pores(results.pores);
        if calibrated
            hist(getappdata(fig,'hist_ax'),[nonedge_pores.EquivDiameter]/pixpernm^2,num_bins);
        else
            hist(getappdata(fig,'hist_ax'),[nonedge_pores.EquivDiameter],num_bins);
        end
        
        status('Updating results summary...');
        % Summary stuff
        str = sprintf('%.2e',results.(statname).poredensity);
        set(res_hs.summary.poredens,'String',str); %pore density
        if calibrated
            set(res_hs.summary.poredensunits,'String','pores/nm^2');
        else
            set(res_hs.summary.poredensunits,'String','pores/pixel^2');
        end
        str = sprintf('%.3g',results.(statname).porosity);
        set(res_hs.summary.porosity,'String',str); % porosity
        
        % Table
        status('Updating results table...');
        table = res_hs.table;
        heading_strings = table.heading_strings;
        for i = 1:length(heading_strings)
            switch heading_strings{i}
                case 'Diameter'
                    val = results.(statname).mean_diam;
                    stdev = results.(statname).std_diam;
                    str = sprintf('%.1f +- %.1f',val,stdev);
                case 'Area'
                    val = results.(statname).mean_area;
                    stdev = results.(statname).std_area;
                    str = sprintf('%.1f +- %.1f',val,stdev);
                case 'Major Axis'
                    val = results.(statname).mean_majax;
                    stdev = results.(statname).std_majax;
                    str = sprintf('%.1f +- %.1f',val,stdev);
                case 'Minor Axis'
                    val = results.(statname).mean_minax;
                    stdev = results.(statname).std_minax;
                    str = sprintf('%.1f +- %.1f',val,stdev);
                case 'Avg Intensity'
                    val = results.(statname).mean_avg_intensity;
                    stdev = results.(statname).std_avg_intensity;
                    str = sprintf('%.1f +- %.1f',val,stdev);
                case 'Roundness'
                    val = results.(statname).mean_roundness;
                    stdev = results.(statname).std_roundness;
                    str = sprintf('%.2f +- %.2f',val,stdev);
                otherwise
                    str = '-';
            end     
            set(table.selpore_hs(i),'String','-');
            set(table.meanpore_hs(i),'String',str);
        end
        %remove currently selected pore outline if present
        if any(ishandle(getappdata(fig,'selpore_h')))
            delete(getappdata(fig,'selpore_h'));
            setappdata(fig,'selpore_h',[]);
        end
        % if pore boundaries are showing, then recalculate them
        view_hs = getappdata(fig,'view_handles');
        if get(view_hs.showbound,'Value');
            pore_processing_gui_call([],[],'pore boundaries');
        end
        % Mark results as current
        pore_processing_gui_call([],[],'set_results_current',true);        
        
        status('Results updated');
        
%% set_results_current
    case 'set_results_current' 
        if isempty(varargin)
            tf = getappdata(fig,'results_current_flag');
        else
            tf = varargin{1};
        end
        results_handles = getappdata(fig,'results_handles');
        calc_b = results_handles.calc_b;
        if tf
            % set results as current (disable update_results button, change to gray)
            set(calc_b,'Enable','off')
            set(calc_b,'ForegroundColor','g');
            set(calc_b,'BackgroundColor',[.94,.94,.94]);

        else % set results as not current
            set(calc_b,'Enable','on')
            set(calc_b,'ForegroundColor','g')
            set(calc_b,'BackgroundColor','r')
            
        end
        setappdata(fig,'results_current_flag',tf);
            
%% change view        
    case 'change view'
        view_hs = getappdata(fig,'view_handles');
        if isempty(varargin)
            sel = get(view_hs.rbg,'SelectedObject');
            if sel==view_hs.orig
                imname = 'orig';
            elseif sel==view_hs.bgc
                imname = 'bgc';
            elseif sel==view_hs.bw
                imname = 'bw';
            end
            pore_processing_gui_call([],[],'change view',imname);
        else
            imname = varargin{1};
            if length(varargin)>1 && varargin{2}
                silent_mode = true;
            else
                silent_mode = false;
            end
            imh = getappdata(fig,'image_h');
            if ~silent_mode, status('Changing view...'), end
            set(imh,'CData',getappdata(fig,imname));
            set(view_hs.(imname),'Value',1);
            set(view_hs.rbg,'SelectedObject',view_hs.(imname));
            if ~silent_mode,status('View changed'); end
        end
%% pore boundaries        
    case 'pore boundaries'
        view_hs = getappdata(fig,'view_handles');
        if isempty(varargin)
            showbound = get(view_hs.showbound,'Value');
        else
            showbound = varargin{1};
        end
        if showbound && ~isempty(getappdata(fig,'results'))
            % show boundaries
            bound_h = getappdata(fig,'bound_h');
            if isempty(bound_h)
                % no existing boundaries
                results = getappdata(fig,'results');
                for i = 1:length(results.pores)
                    if results.pores(i).non_edge_flag
                        %non-edge pore -- red outline
                        b_color = 'r';
                    else
                        %edge pore -- green outline
                        b_color = 'g';
                    end
                    boundary = results.pores(i).boundary;
                    pore_num = results.pores(i).pore_num;
                    bound_h(i) = line(boundary(:,2),boundary(:,1),...
                        'Color',b_color,'Parent',getappdata(fig,'image_ax'),...
                        'ButtonDownFcn',{@pore_processing_gui_call,'porebound click',pore_num});
                end
                setappdata(fig,'bound_h',bound_h);
                status('Showing pore boundaries');
%                 %old way
%                 for i = 1:length(results.B)
%                     bound_h(i) = line(results.B{i}(:,2),results.B{i}(:,1),...
%                         'Color','r','Parent',getappdata(fig,'image_ax'),...
%                         'ButtonDownFcn',{@pore_processing_gui_call,'porebound click',i});
%                 end
%                 setappdata(fig,'bound_h',bound_h);
            else
                % delete and recalculate
                if any(ishandle(bound_h))
                    delete(bound_h);
                end
                setappdata(fig,'bound_h',[]);
                pore_processing_gui_call([],[],'pore boundaries');
            end
        else
            % hide boundaries
            bound_h = getappdata(fig,'bound_h');
            if ~isempty(bound_h)
                if any(ishandle(bound_h))
                    delete(bound_h);
                end
                setappdata(fig,'bound_h',[]);
                status('Removed pore boundaries from display.')
            end
        end
%% pixpernm
    case 'pixpernm'
        res_hs = getappdata(fig,'results_handles');
        calib_hs = res_hs.calib;
        sel = get(calib_hs.calib_rbg,'SelectedObject');
        if sel==calib_hs.calib_none_r
            pixpernm = [];
            status('Calibration removed, only pixel statistics will be calculated');
        elseif sel==calib_hs.calib_manual_r
            pixpernm = str2double(get(calib_hs.calib_manual_e,'String'));
            status(['Calibration set at ' num2str(pixpernm) ' pixels/nm']);
        elseif sel==calib_hs.calib_scope_r
            scopename = popup_str_val(calib_hs.calib_p1);
            mag = popup_str_val(calib_hs.calib_p2);
            pixpernm = get_calibration(scopename,mag);
            status(['Calibration set at ' num2str(pixpernm) ' pixels/nm']);
        end
        
        setappdata(fig,'pixpernm',pixpernm);
        % Mark results as non-current
        pore_processing_gui_call([],[],'set_results_current',false);        
        
        
%% microscope
    case 'microscope'
        % Microscope selection changed, update list of available
        % magnifications
        res_hs = getappdata(fig,'results_handles');
        calib_hs = res_hs.calib;
        scope_name = popup_str_val(calib_hs.calib_p1);
        if strmatch(scope_name,'Microscope')
            % remove all magnifications from the list
            set(calib_hs.calib_p2,'String',{'Magnification'})
        else
            mag_strs = get_magnifications(scope_name);
            set(calib_hs.calib_p2,'String',[{'Magnification'},mag_strs'],'Value',1);
            status('Microscope changed, please select magnification')            
        end
        
%% cutoff
    case 'cutoff'
        % remove pores which match given conditions
        % determine property
        res_hs = getappdata(fig,'results_handles');
        cutoff_hs = res_hs.cutoff;
        prop_name = popup_str_val(cutoff_hs.cutoff_prop_p);
        switch prop_name
            case 'Area'
                property = 'Area';
                units_power = 2;
            case 'Major Axis'
                property = 'MajorAxisLength';
                units_power = 1;
            case 'Minor Axis'
                property = 'MinorAxisLength';
                units_power = 1;
            case 'Diameter'
                property = 'EquivDiameter';
                units_power = 1;
            case 'Avg Intensity'
                property = 'avg_intensity';
                units_power = 0; %not affected by scaling
            case 'Roundness'
                property = 'RoundnessIndex';
                units_power = 0; %not affected by scaling
            otherwise
                error('Unknown property name requested when trying to do a property-based cutoff')
        end
        % determane relation
        relation = popup_str_val(cutoff_hs.cutoff_rel_p);
        switch relation
            case '>'
                rel_fcn = @gt;
            case '<'
                rel_fcn = @lt;
            case '>='
                rel_fcn = @ge;
            case '<='
                rel_fcn = @le;
            case '='
                rel_fcn = @eq;
        end
        % get value
        cutoff_val = str2double(get(cutoff_hs.cutoff_e,'String'));
        % get units
        u = popup_str_val(cutoff_hs.cutoff_units_p);
        % identify matching pores and remove
        r = getappdata(fig,'results');
        pore_vals = [r.pores.(property)];
        switch u
            case 'nm'
                % have to convert cutoff value from nm to pixels (results
                % are always kept in pixels)
                if isempty(r.pixpernm)
                    error('Can''t use nanometer scaling for cutoff when there is no calibration in the results!! No pores removed');
                end
                cutoff_val = cutoff_val*r.pixpernm^units_power;
            case 'pixels'
                % results are already in pixels
            otherwise
                error(['Unknown unit value: ' u]);
        end
        pores_to_remove = find(rel_fcn(pore_vals,cutoff_val));
        status(['Removing pores ' relation ' ' num2str(cutoff_val) ' ' u '...']);
        pore_processing_gui_call([],[],'remove pore',pores_to_remove);
        
%% status        
    case 'status'
        msg = varargin{1};
        status(msg);
    otherwise
        disp(callbackstr)
        
end


%% %%%% Nested functions
%% Status function
    function status(msg)
        status_h = getappdata(fig,'status_handle');
        set(status_h,'String',msg);
        drawnow;
    end

%% Log function
    function addlog(cmd,varargin)
%        log_h = getappdata(fig,'log_handle');
%        alglog = getappdata(fig,'alglog');
%        prevcmds = {alglog{:,1}};
%        
%         %keyboard;
%         switch cmd
%             case 'load'
%                 % reset log and start again with new image
%                 % Format: 'load', image name
%                 imname = varargin{1};
%                 alglog = {'load',imname};               
%             case 'crop'
%                 % reset log after previous cropping (or just put after
%                 % load)
%                 % Format: 'crop', [xmin,xmax,ymin,ymax]
%                 axlim = varargin{1};
%                 % Check for previous cropping events
%                 cropind = find(strcmp('crop',prevcmds),1,'first');
%                 loadind = find(strcmp('load',prevcmds),1,'first');
%                 if cropind
%                    % remove everything after previous cropping
%                    alglog(cropind+1:end) = [];
%                 elseif loadind
%                     % remove everything after previous loading
%                     alglog(loadind+1:end) = [];
%                     
%                 end
%                 alglog{end+1,1:length(varargin+1)} = {'crop',axlim};
%                 
%             case 'bgc'
%                 % reset log after last cropping
%                 % Format: 'bgc', filter size
%                 filtersize = varargin{1};
%                 % Check for previous cropping events
%                 cropind = find(strcmp('crop',prevcmds),1,'last');
%                 loadind = find(strcmp('load',prevcmds),1,'last');
%                 if cropind
%                    % remove everything after previous cropping
%                    alglog(cropind+1:end) = [];
%                 elseif loadind
%                     % remove everything after previous loading
%                     alglog(loadind+1:end) = [];
%                     
%                 end
% 
%                 
%             case 'se'
%             case 'calc' % calculate results
%                 % remove any prior pore removals and calculations
%             case 'remove'
%                 
%             otherwise
%         end
%         
%         setappdata(fig,'alglog',alglog);
%         showlog(alglog);
    end
    

%% popup_str_val Function    
    function [chosen_str,val] = popup_str_val(popup_handle)
    % returns the chosen string and value for a popupmenu, given the handle
    % for it
        str_list = get(popup_handle,'String');
        val = get(popup_handle,'Value');
        chosen_str = str_list{val};
    end
    
%% get_calibration
    function pixpernm = get_calibration(scope_name,mag_str)
        calib_info = parse_calibrations_file;
        [mag_strs,pixpernm_list] = get_magnifications(scope_name,calib_info);

        mag_ind = find(strcmp(mag_str,mag_strs), 1);
        if isempty(mag_ind)
            status(['Error: Magnification ''' mag_str ''' not found in MicroscopeCalibrations.txt for microscope ''' scope_name '''; Calibration not set']);
            error(['Magnification ''' mag_str ''' not found in MicroscopeCalibrations.txt for microscope ''' scope_name '''; Calibration not set']);
        end

        pixpernm = pixpernm_list{mag_ind};

    end
%% get_magnifications
    function [mag_strs,pixpernm_list,calib_info] = get_magnifications(scope_name,calib_info)
        if nargin<2
            calib_info = parse_calibrations_file;
        end
        
        scope_names = calib_info(:,1);
        name_ind = find(strcmp(scope_name,scope_names), 1);
        if isempty(name_ind)
            status(['Error: Microscope name ''' scope_name ''' not found in MicroscopeCalibrations.txt; Calibration not set']);
            error(['Microscope name ''' scope_name ''' not found in MicroscopeCalibrations.txt; Calibration not set']);
        end

        mag_strs = calib_info{name_ind,2}(:,1);
        pixpernm_list = calib_info{name_ind,2}(:,2);
        
    end

        
            
end %end of outer main function
%% %%%% Subfunctions
%% Find parent figure
function fig = getfig(h)

if ishandle(h)
    if strcmp(get(h,'Type'),'Figure')
        fig = h;
        return
    end    
    p = get(h,'Parent');
    if strcmp(get(p,'Type'),'Figure');
        fig = p;
    else
        fig = getfig(p);
    end
else
    fig = [];
end
end
%% Calculate pore statistics
function res = calc_stats(res,pixpernm)
if nargin<2
    pixpernm = []; %default is pixel stats
end
if isempty(pixpernm)
    statnames = {'stats'};
    res.pixpernm = [];
    pixpernm = 1; % calculating pixel stats
else
    statnames = {'stats','nmstats'}; % calculate both pixel and nm stats
    res.pixpernm = pixpernm;
    pixpernm = [1,pixpernm]; 
end

nonedge_pores = find_nonedge_pores(res.pores);


for i = 1:length(statnames)
    statname = statnames{i};
    ppn = pixpernm(i);
    diams = [nonedge_pores.EquivDiameter];
    res.(statname).mean_diam = mean(diams)/ppn;
    res.(statname).std_diam = std(diams)/ppn;
    res.(statname).max_diam = max(diams)/ppn;
    areas = [nonedge_pores.Area];
    res.(statname).mean_area = mean(areas)/ppn^2;
    res.(statname).std_area = std(areas)/ppn^2;
    res.(statname).max_area = max(areas)/ppn^2;
    majax = [nonedge_pores.MajorAxisLength];
    res.(statname).mean_majax = mean(majax)/ppn;
    res.(statname).std_majax = std(majax)/ppn;
    res.(statname).max_majax = max(majax)/ppn;
    minax = [nonedge_pores.MinorAxisLength];
    res.(statname).mean_minax = mean(minax)/ppn;
    res.(statname).std_minax = std(minax)/ppn;
    res.(statname).max_minax = max(minax)/ppn;
    avg_intensities = [res.pores.avg_intensity]; % use all pores, not just non-edge
    res.(statname).mean_avg_intensity = mean(avg_intensities);
    res.(statname).std_avg_intensity = std(avg_intensities);
    roundnesses = [res.pores.RoundnessIndex];
    res.(statname).mean_roundness = mean(roundnesses);
    res.(statname).std_roundness = std(roundnesses);
    
    res.(statname).processed_area = numel(res.L)/ppn^2;
    num_edge_pores = length(res.pores)-length(nonedge_pores);
    res.(statname).poredensity = (length(nonedge_pores)+num_edge_pores/2) / res.(statname).processed_area; %number of pores/processed area (edge pores count as 1/2)
    res.(statname).porosity = sum([res.pores.Area])/ppn^2/res.(statname).processed_area; % pore area/processed area (unitless)
end
end
%% find_nonedge_pores
function nep = find_nonedge_pores(pores_structure)
nep = pores_structure(logical([pores_structure.non_edge_flag]));
end
%% find_edge_pores
function edge_pores = find_edge_pores(pores_structure)
edge_pores = pores_structure(~logical([pores_structure.non_edge_flag]));
end
