function [calib_info] = parse_calibrations_file(fname)
% Parses the requested calibrations file 

if nargin<1
    fname = 'MicroscopeCalibrations.txt';
end

fh = fopen(fname,'r');

scope_counter = 0;
mag_counter = 0;
line_counter = 0;
scope_names = {};
mags = {};

while 1
    line = fgetl(fh);
    line_counter = line_counter+1;
    if ~ischar(line)
        % reached end of file
        break
    end
    % Clean up line
    line = regexprep(line,'\s*:\s*',' : '); % reduce multiple spaces around : to 1
    line = regexprep(line,'^\s*|\s*$',''); %remove leading and trailing spaces
    
    if isempty(line)
        continue
    end

    if regexp(line,'^%')
        % if first non-space character is %, then ignore the line
        continue
    elseif regexp(line,':')
        % magnification line
        mag_counter = mag_counter+1;
        if length(strfind(line,':'))>1
            error(['Error at line ' num2str(line_counter) ' in microscope calibration file ''' fname '''. ' sprintf('\n') 'There can be at most one colon per line in this file.']);
        end
        [tokens,match] = regexp(line,'^\s*(.+)\s*:\s*(.+)\s*$','tokens','match');
        if scope_counter==0
            error(['Error at line ' num2str(line_counter) ' in microscope calibration file ''' fname '''. ' sprintf('\n') 'Magnification line encountered before microscope name specified.']);
        elseif isempty(match) || length(tokens{1})~=2
            error(['Error at line ' num2str(line_counter) ' in microscope calibration file ''' fname '''. ' sprintf('\n') 'Magnification line did not parse correctly.']);
        else
            mag_str = tokens{1}{1};
            pixpernm = str2double(tokens{1}{2});
            if isnan(pixpernm)
                error(['Error at line ' num2str(line_counter) ' in microscope calibration file ''' fname '''.' sprintf('\n') 'Pixels/nm value ''' tokens{1}{2} ''' failed to convert to a numerical value.']);
            end               
            mags{scope_counter}{mag_counter,1} = tokens{1}{1};
            mags{scope_counter}{mag_counter,2} = pixpernm;
        end
        
    else
        % scope name line
        if scope_counter~=0 && mag_counter==0 
            warning(['No magnifications supplied for microscope named ' scope_names{scope_counter} ' in microscope calibration file ''' fname '''.']); 
        end
        scope_counter = scope_counter+1;
        mag_counter = 0;
        scope_names{scope_counter} = line;
        mags{scope_counter} = {};
    end
end
if isempty(mags{scope_counter})
    warning(['No magnifications supplied for microscope named ' scope_names{scope_counter} ' in microscope calibration file ''' fname '''.']); 
end

calib_info= cell(length(scope_names));
calib_info = [scope_names; mags]';



