function histFromResults(results,numbins,propertyname)
% Plots a histogram of pore data based on a results structure generated
% by Pore Image Processor. The user can supply a results structure as the
% first input, the number of desired histgram bins as the second input, and
% the desired pore property to plot as the third input, or can leave any
% inputs empty to use defaults (prompt user to select results file; 50
% bins; pore equivalent diameters)
%
% If specified, property name must be one of 'diameter', 'area', 'major
% axis', or 'minor axis'.  
%
% Histogram data units are nanometers if the data is calibrated, pixels
% otherwise. The units are indicated in the x-axis label
%
% USAGES: 
%   histFromResults;
% With no inputs, the user is prompted to select a results file, and a
% histogram of pore equivalent diameters is plotted with 50 bins.
%
%   histFromResults([],30)
% Same as previous, except the histogram has only 30 bins.
%
%   histFromResults([],[],'area')
% Prompts the user to select a results file, then plots a histogram of pore
% areas (rather than the default diameters) with the default 50 bins.
%
%   histFromResults([],100,'major axis')
% Prompts user to select a results file, then plots a histogram of pore
% major axes with 100 bins. 
%
%   histFromResults(res)
% Used the results structure 'res' for data (rather than prompting the user
% to choose a file) and plots a histogram of pore equivalent diameters with
% 50 bins. 
%
%   histFromResults(res,75,'area')
% Same as previous, except 75 bins in the histogram, and the data is pore
% areas rather than diameters.


% Set default property name
if nargin<3 || isempty(propertyname)
    propertyname = 'diameter';
end

% Set default number of histogram bins
if nargin<2 || isempty(numbins)
    numbins = 50;
end

% load results from file if not specified
if nargin<1 || isempty(results)
    %Load results from file
    [fname,pname] = uigetfile('*_r.mat','Load Results File');
    r = load([pname fname],'results');
    results = r.results;
end

% Determine fieldname from propertyname
switch lower(propertyname)
    case 'diameter'
        fieldname = 'EquivDiameter';
        propertylabel = 'Pore Equivalent Diameter';
        exponent = 1;
    case 'area'
        fieldname = 'Area';
        propertylabel = 'Area';
        exponent = 2;
    case 'major axis'
        fieldname = 'MajorAxisLength';
        propertylabel = 'Major Axis';
        exponent = 1;
    case 'minor axis'
        fieldname = 'MinorAxisLength';
        propertylabel = 'Minor Axis';
        exponent = 1;        
    case 'avg intensity'
        fieldname = 'avg_intensity';
        propertylabel = 'Avg Intensity';
        exponent = 1;
    otherwise
        error(['Unknown property name: ' propertyname ]);
end

% Make a new figure
figure;

% Make the histogram
if isempty(results.pixpernm)
    hist([results.pores.(fieldname)],numbins);
    xlabel([propertylabel ' (pixels)']);
else
    hist([results.pores.(fieldname)]/results.pixpernm^exponent,numbins);
    xlabel([propertylabel ' (nm)']);
end
    
ylabel('Number of Pores');