function h = arrow(x,y,varargin)
% This function arrow should work much like the MATLAB function LINE, but add 
% arrowheads.  Give a vector or matrix of xs (if a matrix, connected points 
% should be in columns) and corresponding ys. The arrow head will go on the
% far end of the line.
% The default size of the arrow heads is 1/10th the length of the arrow.  
% To change the length factor to 0.2 and the arrowhead angle to pi/6, the syntax is 
% h = arrow(x,y,'ArrowheadLengthFactor',.2,'ArrowheadAngle',pi/6)
% You can also choose between two arrowhead types, 'line' or 'solid' by
% specifying the 'ArrowheadType' property.  
%
% another example of syntax
% h = arrow(xs,ys,{'LineWidth','LineStyle'}',{1,':';2,'-';3,'--'},'ArrowheadAngle',[pi/6,pi/2,pi/4],'Arrowheadlengthfactor',[0.5,.1,.2])
% this uses the cell array version of set to set the three arrows specified
% in xs and ys to be different widths and different styles.  Notice also
% that the angle and length factor can also be different for all three

holdstate = get(gca,'NextPlot');
length_factor = 1/10;
arrowhead_angle = pi/4;
arrowhead_type = 'line'; % other option is solid

% Check inputs for changes to length factor or arrowhead angle or arrowhead type
ch_lengthfactor = find(strcmpi(varargin,{'ArrowheadLengthFactor'}));
if ~isempty(ch_lengthfactor)
    length_factor = varargin{ch_lengthfactor+1};
    varargin(ch_lengthfactor:ch_lengthfactor+1) = [];
end
ch_angle = find(strcmpi(varargin,{'ArrowheadAngle'}));
if ~isempty(ch_angle)
    arrowhead_angle = varargin{ch_angle+1};
    varargin(ch_angle:ch_angle+1) = [];
end
ch_type = find(strcmpi(varargin,'ArrowheadType'));
if ~isempty(ch_type)
    arrowhead_type = varargin{ch_type+1};
    varargin(ch_type:ch_type+1) = [];
end

if size(x,1)==1
    x = x';
    y = y';
end

% Calculate length of lines
diff_x = diff(x,[],1);
diff_y = diff(y,[],1);
len = sum(sqrt(diff_x.^2+diff_y.^2),1);

% Length arrowheads
len_arrow = len.*length_factor(:)';

% find points for end of arrowhead segments
% Idea is to find last line segment, then make arrowhead point in that
% direction

% find angle last line segment points at 
angle = atan2(diff_y(end,:),diff_x(end,:));

% get points
arrowhead_angle = arrowhead_angle(:)'; %make sure it's a row vector
angle1 = angle+pi+(arrowhead_angle/2);
angle2 = angle1-arrowhead_angle;
pts1_x = x(end,:)+len_arrow.*cos(angle1); 
pts1_y = y(end,:)+len_arrow.*sin(angle1);
pts2_x = x(end,:)+len_arrow.*cos(angle2); 
pts2_y = y(end,:)+len_arrow.*sin(angle2);

% set up arrowhead lines
% start at pts1 go to end of line, then to pts2
arrowhead_xs = [pts1_x; x(end,:); pts2_x];
arrowhead_ys = [pts1_y; y(end,:); pts2_y];
%h2 = fill(arrowhead_xs,arrowhead_ys);
if strcmpi(arrowhead_type,'solid')
    h = line(x,y);
    if ~isempty(varargin)
        set(h,varargin{:});
    end
    c = get(h,'Color');
    if length(h)==1
        hold on;
        h2 = fill(arrowhead_xs,arrowhead_ys,c);
    else
        for i = 1:length(c)
            hold on;
            h2(i) = fill(arrowhead_xs(:,i),arrowhead_ys(:,i),c{i});
        end
    end
    h = [h,h2(:)];
else %default type is 'line'
    xa = [x;repmat(NaN,[1,size(x,2)]);arrowhead_xs];
    ya = [y;repmat(NaN,[1,size(x,2)]);arrowhead_ys];

    h = line(xa,ya);
    if ~isempty(varargin)
        set(h,varargin{:});
    end
end
set(gca,'NextPlot',holdstate);
